<?php
/**
 * Simplified Pro functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Simplified_Lite
 */

if ( ! function_exists( 'simplified_lite_setup' ) ) :

	// Set the default content width.
		$GLOBALS['content_width'] = 1024;
		
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function simplified_lite_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on Simplified Pro, use a find and replace
		 * to change 'simplified-lite' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'simplified-lite', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		// Enable support for Post Thumbnails on posts and pages.
		add_theme_support( 'post-thumbnails' );
		
		// create recent posts thumbnails
		add_image_size( 'simplified-recent', 100, 75, true );
		
		// create small wp gallery thumbnails
		if( esc_attr(get_theme_mod( 'simplified_lite_widget_gallery_thumbnails', false ) ) ) :
		add_image_size( 'simplified-widget-gallery', 100, 100, true );		
		endif;

		// create featured images for the default blog style
		if( esc_attr(get_theme_mod( 'simplified_lite_default_thumbnails', false ) ) ) :
			add_image_size( 'simplified-default', 650, 400, true );
		endif;

		// create featured images for the list blog style with sidebars
		if( esc_attr(get_theme_mod( 'simplified_lite_list_sidebar_thumbnails', false ) ) ) :
			add_image_size( 'simplified-list-sidebar', 300, 360, true );
		endif;
		
		// create featured images for the centered blog style
		if( esc_attr(get_theme_mod( 'simplified_lite_centered_thumbnails', false ) ) ) :
			add_image_size( 'simplified-centered', 1024, 600 );
		endif;		
	
		// create featured images for the single post thumbnail
		if( esc_attr(get_theme_mod( 'simplified_lite_singlepost_thumbnails', false ) ) ) :
			add_image_size( 'simplified-singlepost', 1024, 600, true );
		endif;
	
		
		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'primary' => esc_html__( 'Primary', 'simplified-lite' ),
			'footer' => esc_html__( 'Footer', 'simplified-lite' ),
			'social' => esc_html__( 'Social', 'simplified-lite' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );
		
		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'simplified_lite_custom_background_args', array(
			'default-color' => 'f2f2f2',
			'default-image' => '',
		) ) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );
		
		// Add support for responsive embedded content.
		add_theme_support( 'responsive-embeds' );
		
		// Add support for full and wide align images.
		add_theme_support( 'align-wide' );
		
		// Add support for editor styles.
		add_theme_support( 'editor-styles' );

		// Enqueue editor styles.
		add_editor_style( array( '/css/classic-editor.css', simplified_lite_fonts_url() ) );	
		
		/**
		 * Add support for core custom logo.
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support( 'custom-logo', array(
			'flex-width'  => true,
			'flex-height' => true,
		) );
	}
endif;
add_action( 'after_setup_theme', 'simplified_lite_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 * Priority 0 to make it available to lower priority callbacks.
 * @global int $content_width
 */
function simplified_lite_content_width() {
	$content_width = $GLOBALS['content_width'];
	// Check if the page has a sidebar.
	if ( is_active_sidebar( 'left-sidebar'  ) || is_active_sidebar( 'right-sidebar' ) || is_active_sidebar( 'blog-sidebar' ) ) {
		$content_width = 640;
	}	
  $GLOBALS['content_width'] = apply_filters( 'simplified_lite_content_width', $content_width );
}
add_action( 'template_redirect', 'simplified_lite_content_width', 0 );


/**
 * Handles JavaScript detection.
 * Adds a `js` class to the root `<html>` element when JavaScript is detected.
 */
function simplified_lite_javascript_detection() {
	echo "<script>(function(html){html.className = html.className.replace(/\bno-js\b/,'js')})(document.documentElement);</script>\n";
}
add_action( 'wp_head', 'simplified_lite_javascript_detection', 0 );


/**
 * Register Google Fonts.
 */
if ( ! function_exists( 'simplified_lite_fonts_url' ) ) :

function simplified_lite_fonts_url() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';

	// Translators: If there are characters in your language that are not supported by Noto Serif, translate this to 'off'. Do not translate into your own language.
	if ( 'off' !== _x( 'on', 'Oswald font: on or off', 'simplified-lite' ) ) {
		$fonts[] = 'Oswald:400,500';
	}		

	// Translators: To add an additional character subset specific to your language, translate this to 'greek', 'cyrillic', 'devanagari' or 'vietnamese'. Do not translate into your own language.
	$subset = _x( 'no-subset', 'Add new subset (greek, cyrillic, devanagari, vietnamese)', 'simplified-lite' );

	if ( 'cyrillic' == $subset ) {
		$subsets .= ',cyrillic,cyrillic-ext';
	} elseif ( 'greek' == $subset ) {
		$subsets .= ',greek,greek-ext';
	} elseif ( 'devanagari' == $subset ) {
		$subsets .= ',devanagari';
	} elseif ( 'vietnamese' == $subset ) {
		$subsets .= ',vietnamese';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'subset' => urlencode( $subsets ),
		), 'https://fonts.googleapis.com/css' );
	}

	return esc_url_raw( $fonts_url );
}
endif;


/**
 * Add preconnect for Google Fonts.
 * @param array  $urls           URLs to print for resource hints.
 * @param string $relation_type  The relation type the URLs are printed.
 * @return array $urls           URLs to print for resource hints.
 */
function simplified_lite_resource_hints( $urls, $relation_type ) {
	if ( wp_style_is( 'simplified-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
		$urls[] = array(
			'href' => 'https://fonts.gstatic.com',
			'crossorigin',
		);
	}

	return $urls;
}
add_filter( 'wp_resource_hints', 'simplified_lite_resource_hints', 10, 2 );

/**
 * Enqueue scripts and styles.
 */
function simplified_lite_scripts() {

	
	// Enable or Disable Google default fonts
	if( esc_attr(get_theme_mod( 'simplified_lite_default_google_fonts', true ) ) ) {
		wp_enqueue_style( 'simplified-fonts', simplified_lite_fonts_url(), array(), null );
	}
	
	// Icons
	wp_enqueue_style( 'icomoon', get_template_directory_uri() . '/css/icons.css', array(), '2018' );
	
	// Theme block front-end stylesheet.
    if ( false == esc_attr(get_theme_mod( 'simplified_lite_block_css', false ) ) ) {
	wp_enqueue_style( 'simplified-block-style', get_template_directory_uri() . '/css/blocks.css', array(), null );
    }	
	
	// Stylesheets
	wp_enqueue_style( 'simplified-style', get_stylesheet_uri() );

	// Scripts
	wp_enqueue_script( 'simplified-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20151215', true );
		
	// Theme functions and navigation
	wp_enqueue_script( 'simplified-theme', get_template_directory_uri() . '/js/theme.js', array(), '20151215', true );

	$screen_reader_text = array(
		'expand'   => esc_html__( 'expand child menu', 'simplified-lite' ),
		'collapse' => esc_html__( 'collapse child menu', 'simplified-lite' ),
	);
	$script_data = 'var screenReaderText = ' . json_encode( $screen_reader_text ) . ';';
	wp_register_script( 'simplified-script', get_template_directory_uri() . '/js/navigation.js', array( 'jquery' ), '20160816', true );
	wp_add_inline_script( 'simplified-script', $script_data, 'before' );
	wp_enqueue_script( 'simplified-script' );

	// Comments 
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'simplified_lite_scripts' );


 /*--------------------------------------------------------
 Enqueue Gutenberg editor styles
 -------------------------------------------------------*/
function simplified_lite_gutenberg_editor_styles() { 
	wp_enqueue_style( 'simplified-editor-styles', get_template_directory_uri() . '/css/block-editor.css'); 
}
// only name the function and leave the enqueue as enqueue_block_editor_assets
add_action( 'enqueue_block_editor_assets', 'simplified_lite_gutenberg_editor_styles' );

if ( ! function_exists( 'simplified_lite_add_gutenberg_features' ) ) {
	function simplified_lite_add_gutenberg_features() {
		
		/* Gutenberg Colour Palette */
		$accent_color = get_theme_mod( 'simplified_lite_accent_colour' ) ? get_theme_mod( 'simplified_lite_accent_colour' ) : '#bb7514';

		add_theme_support( 'editor-color-palette', array(
			array(
				'name' 	=> _x( 'Accent', 'Name of the accent color in the Gutenberg palette', 'simplified-lite' ),
				'slug' 	=> 'accent',
				'color' => esc_attr($accent_color),
			),		
			array(
				'name' 	=> _x( 'Dark', 'Name of the dark grey, almost black color in the Gutenberg palette', 'simplified-lite' ),
				'slug' 	=> 'dark',
				'color' => '#1a1a1a',
			),

			array(
				'name' 	=> _x( 'Grey', 'Name of the grey color in the Gutenberg palette', 'simplified-lite' ),
				'slug' 	=> 'grey',
				'color' => '#9b9b9b',
			),
			array(
				'name' 	=> _x( 'White', 'Name of the white color in the Gutenberg palette', 'simplified-lite' ),
				'slug' 	=> 'white',
				'color' => '#fff',
			),
		) );
		
		/* Gutenberg Font Sizes */

		add_theme_support( 'editor-font-sizes', array(
			array(
				'name' 		=> _x( 'Small', 'Name of the small font size in Gutenberg', 'simplified-lite' ),
				'shortName' => _x( 'S', 'Short name of the small font size in the Gutenberg editor.', 'simplified-lite' ),
				'size' 		=> 16,
				'slug' 		=> 'small'
			),
			array(
				'name' 		=> _x( 'Regular', 'Name of the regular font size in Gutenberg', 'simplified-lite' ),
				'shortName' => _x( 'M', 'Short name of the regular font size in the Gutenberg editor.', 'simplified-lite' ),
				'size' 		=> 18,
				'slug' 		=> 'regular'
			),
			array(
				'name' 		=> _x( 'Large', 'Name of the large font size in Gutenberg', 'simplified-lite' ),
				'shortName' => _x( 'L', 'Short name of the large font size in the Gutenberg editor.', 'simplified-lite' ),
				'size' 		=> 24,
				'slug' 		=> 'large'
			),
			array(
				'name' 		=> _x( 'Larger', 'Name of the larger font size in Gutenberg', 'simplified-lite' ),
				'shortName' => _x( 'XL', 'Short name of the larger font size in the Gutenberg editor.', 'simplified-lite' ),
				'size' 		=> 32,
				'slug' 		=> 'larger'
			)
		) );		
	
		}
	add_action( 'after_setup_theme', 'simplified_lite_add_gutenberg_features' );

}

// Include better comments file
require get_template_directory() .'/inc/comment-style.php';

// Theme info page class
require get_template_directory() . '/inc/about/simplified-lite-class-about.php';
	
// Theme Info Page
require get_template_directory() . '/inc/about/simplified-lite-info.php';

// Get recent posts widget
require get_template_directory() . '/inc/widgets/class-simplified-recent-posts-widget.php';

// Implement the Custom Header feature.
require get_template_directory() . '/inc/sidebars.php';

// Custom template tags for this theme.
require get_template_directory() . '/inc/template-tags.php';

// Functions which enhance the theme by hooking into WordPress.
require get_template_directory() . '/inc/template-functions.php';

// Customizer additions.
require get_template_directory() . '/inc/customizer.php';

// Load CSS overrides
require get_template_directory() . '/inc/inline-styles.php';

// Load Jetpack compatibility file.
if ( defined( 'JETPACK__VERSION' ) ) {
	require get_template_directory() . '/inc/jetpack.php';
}
